/********************************************************************
 * (C) Copyright 1998 by Hewlett-Packard GmbH. All rights reserved. *
 ********************************************************************/

/***********************************************************************
 * File name     : session.c
 ************************************************************************
 * Created by    : Joerg Weedermann, BID R&D, 05.06.96
 * Last modified : ask clearcase
 ************************************************************************
 * File contents :
 * routines to organize the connection and some more management functions
 ************************************************************************
*/

/* ** Segmentation for DOS **
 * MS allows us to put const data into its own named segment (data_seg()).
 * Borland doesn't like that but does allow us to pack strings
 * into the code segment (option -dc).
 */

#if defined(_MSC_VER)
# pragma data_seg("session_c","FAR_DATA")
#elif defined(__BORLANDC__)
# pragma option -dc
#endif /* _MSC_VER etc. */


/* *********************************************************************
   Connections and opening. This functions do not exist in firmware
   ********************************************************************* */

#ifdef BEST_FIRMWARE

#include <capitype.h>
#include <addrmap.h>
#include <hal.h>

#else

#include <typedefs.h>

#include <regconst.h>
#include <dynamic.h>
#include "errcapi.h"

#include "b_io.h"
#include "b_cmd.h"
#include "b_serial.h"
#include "pci.h"
#include "hif.h"
#include "timeout.h"
#include "identos.h"


#if defined(WIN32) || defined(WIN64)
#   include "pci32reg.h"
#endif

#endif

#include <iocommon.h>
#include <propdefs.h>
#include <regxdir.h>
#include <regx10.h>
#include <regx11.h>
#include <session.h>
#include <version.h>


#ifndef BEST_FIRMWARE

/* SCR; 30.10.97; defined here now */
b_int32 GlobalBaseAddr;


/* -------------------------------------------------------------------------
 * handle_array is the global array that holds all information necessary to
 * access hardware and to connect to all needed ports.
 * The elements in the b_handlestype struct are:
 *
 * port, portnumber, entered_port, is_open, is_connected, is_reserved
 * hwinfo, param (RS232 => baudrate), regwidth, regindex, timeouts,
 * 4 function ptrs, dynamic capability list ptr, perfboard_gapmode, lasterr
 *
 * -------------------------------------------------------------------------
 */

#define LAST_ERR_INIT   {{0}, {0}, B_E_BAD_HANDLE, {0}}

#define HANDLE_INIT \
  {B_PORT_RS232, INVALID_OS_HANDLE, B_PORT_COM1, 0, 0, 0, \
  B_HW_INFO_UNKNOWN, 9600UL, 0, REG_INDX_2926, NO_TIMEOUTS, \
  NULL, NULL, NULL, NULL, NULL, 0, LAST_ERR_INIT}

#define HANDLE_INIT_ERR \
  {B_PORT_OFFLINE, INVALID_OS_HANDLE, 0, 0, 0, 0, \
  B_HW_INFO_UNKNOWN, 0, 0, REG_INDX_2925, NO_TIMEOUTS, \
  NULL, NULL, NULL, NULL, NULL, 0, LAST_ERR_INIT}

#define FOUR_H_INITS HANDLE_INIT, HANDLE_INIT, HANDLE_INIT, HANDLE_INIT

b_handlestype handle_array[MAXHANDLES + 1] = {
  FOUR_H_INITS, FOUR_H_INITS, FOUR_H_INITS, FOUR_H_INITS,
  FOUR_H_INITS, FOUR_H_INITS, FOUR_H_INITS, FOUR_H_INITS,
  FOUR_H_INITS, FOUR_H_INITS, FOUR_H_INITS, FOUR_H_INITS,
  FOUR_H_INITS, FOUR_H_INITS, FOUR_H_INITS, FOUR_H_INITS,
  FOUR_H_INITS, FOUR_H_INITS, FOUR_H_INITS, FOUR_H_INITS,
  FOUR_H_INITS, FOUR_H_INITS, FOUR_H_INITS, FOUR_H_INITS,
  FOUR_H_INITS, FOUR_H_INITS, FOUR_H_INITS, FOUR_H_INITS,
  FOUR_H_INITS, FOUR_H_INITS, FOUR_H_INITS, FOUR_H_INITS,

  /* this next entry (at index == MAXHANDLES) is used to init a user's handle
   * to MAXHANDLES so that we can detect an error if the open fails but they
   * don't check the return value. */
  HANDLE_INIT_ERR
};
/* to reset any element of the handle array to initial values... */
b_handlestype handle_null = HANDLE_INIT;


#define B_DEF_CONNECT_TIMEOUT	1000

#define FOUR_TO_INITS   B_DEF_CONNECT_TIMEOUT, B_DEF_CONNECT_TIMEOUT, \
                        B_DEF_CONNECT_TIMEOUT, B_DEF_CONNECT_TIMEOUT

b_int32 connect_timeout_array[MAXHANDLES + 1] = {
  FOUR_TO_INITS, FOUR_TO_INITS, FOUR_TO_INITS, FOUR_TO_INITS,
  FOUR_TO_INITS, FOUR_TO_INITS, FOUR_TO_INITS, FOUR_TO_INITS,
  FOUR_TO_INITS, FOUR_TO_INITS, FOUR_TO_INITS, FOUR_TO_INITS,
  FOUR_TO_INITS, FOUR_TO_INITS, FOUR_TO_INITS, FOUR_TO_INITS,
  FOUR_TO_INITS, FOUR_TO_INITS, FOUR_TO_INITS, FOUR_TO_INITS,
  FOUR_TO_INITS, FOUR_TO_INITS, FOUR_TO_INITS, FOUR_TO_INITS,
  FOUR_TO_INITS, FOUR_TO_INITS, FOUR_TO_INITS, FOUR_TO_INITS,
  FOUR_TO_INITS, FOUR_TO_INITS, FOUR_TO_INITS, FOUR_TO_INITS,
  B_DEF_CONNECT_TIMEOUT
};


/* statics */
static b_errtype BestIdHardware(b_handletype handle, b_OpenType task);


void BestConnectTimeoutSet(b_handletype handle, b_int32 ms)
{
  assert(handle < MAXHANDLES);
  connect_timeout_array[handle] = ms;
}


void BestConnectTimeoutDefSet(b_handletype handle)
{
  assert(handle < MAXHANDLES);
  connect_timeout_array[handle] = B_DEF_CONNECT_TIMEOUT;
}


/*---------------------------------------------------------
* oscheck checks the different operating systems and ensures that
* you do not run best under the wrong os
*---------------------------------------------------------*/
b_ccharptrtype EXPORT OSCheck()
{
  B_OS_VERSION actualOS, targetOS;
  actualOS = GetOsVersion(NULL, &targetOS);

  if (actualOS != targetOS)
  {
    if (targetOS == OS_WINNT && actualOS == OS_WIN95)
      return NULL;
    if (targetOS == OS_WIN95 && actualOS == OS_WINNT)
      return (char *) "This version of best.exe was compiled for WIN95. It will not run under WINNT. Please install the WINNT version of best.exe.";
    else
      return (char *) "This version of best.exe was compiled for another operating system than it is running under. Please use the correct software version.";
  }

  return NULL;
}

/* HL, 15.9.1999: BestDevIdentifierGet moved to mailbox.c */

/* --------------------------------------------------------------------------
 * This helper function is separated because it's used by BestOpenAdv()
 * and one other function.
 * -------------------------------------------------------------------------- */

b_errtype BestFirstFreeHandleGet(b_handletype * pHandle)
{
  b_handletype handle;

  for (handle = 0; handle < (b_handletype) MAXHANDLES; handle++)
  {
    if ((handle_array[handle].is_open == 0) &&
      (handle_array[handle].is_reserved == 0))
      break;
  }

  *pHandle = handle;

  /* no more free handles left ? */
  return(handle == MAXHANDLES ? B_E_NO_HANDLE_LEFT : B_E_OK);
}


/******************************************************************************/
/* --------------------------------------------------------------------------
 * Returns B_E_OK and fBoardWasReset = 1 if the board was reset since Opening
 * OR since the last call to BestBasicIsBoardReset().
 * fBoardWasReset is undefined if the return is NOT B_E_OK.
 * -------------------------------------------------------------------------- */

b_errtype EXPORT BestBasicIsBoardReset(
    b_handletype handle,
    b_int32 *fBoardWasReset)
{
  B_DECLARE_FUNCNAME("BestBasicIsBoardReset");

  b_errtype err;
  B_FCT_PARAM_NULL_POINTER_CHK(fBoardWasReset); 
  

  if (Best16BitRegisterFile(handle))
  {
    b_int32 isReset;
    b_int8 out_zw[OUT_RESET_CHECK];
    b_int16 outsize = OUT_RESET_CHECK;
  
    err = BestBasicCommand(handle, CMD_RESET_CHECK,
			   NULL, IN_RESET_CHECK,
			   out_zw, &outsize);
    (void) BestStream2Long(&isReset, out_zw, 1);
    *fBoardWasReset = isReset != 0 ? 1 : 0;
  }
  else
  {
    b_int32 dwData;
    /* RESET_DETECT_VALUE was written into DUMMY_REGISTER when the port was
     * opened...see if it's still there */
    err = BestDummyGet(handle, &dwData);
    B_ERRCHECK(err);

    if (dwData == RESET_DETECT_VALUE) /*Has NOT been reset since opening */
    {
      *fBoardWasReset = 0;
      B_ERRETURN (B_E_OK);
    }
    else
    {
      *fBoardWasReset = 1;

      /* was reset...write special value back into dummy register */
      dwData = RESET_DETECT_VALUE;
      err = BestDummySet(handle, dwData);
      B_ERRCHECK(err);
    }
  }
  
  B_ERRETURN(err);
}


/* -------------------------------------------------------------------------*/
/* This is the classic open call, called from a user program or from the
 * GUI. It is mapped onto BestOpenAdv to have only one function that handles
 * all open accesses. */
/* -------------------------------------------------------------------------*/

b_errtype EXPORT BestOpen(
    b_handletype * pHandle,
    b_porttype port,
    b_int32 portnum)
{
  /* Do not use B_ERRETURN here !! CZ */
  return BestOpenAdv(pHandle, port, portnum, NULL, B_OPEN_STANDARD);
}
/* --------------------------------------------------------------------------
 * Use this open call.  It is more advanced and can handle several different
 * types of open (with or without license, online or offline) and can even
 * take a license number to enable licenses within the firmware.
 * -------------------------------------------------------------------------- */

b_errtype EXPORT BestOpenAdv(
    b_handletype * pUsrHandle,
    b_porttype port,
    b_int32 portnum,
    b_int32 * license_code,
    b_OpenType task)
{
  B_DECLARE_FUNCNAME("BestOpenAdv");

  B_TRY_VARS;
  b_portnumtype OsHandle = INVALID_OS_HANDLE;
  b_handletype handle = INVALID_BEST_HANDLE;
  b_int32 HPSlotId = 0;         /* used for PCI and FastHIF */
  int j;

  B_TRY_BEGIN
  {
    B_TRY_FCT_PARAM(1, port > B_PORT_OFFLINE);
    /* Bios switch task looks whether *pUsrHandle is a reserved handle. If
     * this is the case it keeps the same handle, otherwise it follows the
     * standard procedure */
    if ((task == B_OPEN_BIOS_SWITCH) || (task == B_OPEN_STANDARD_RESERVED))
    {
      handle = *pUsrHandle;

      if (handle < MAXHANDLES)
      {
        if ((!handle_array[handle].is_open) &&
          handle_array[handle].is_reserved)
        {
          goto handle_label;
        }
      }
    }

    /* MAXHANDLES is a valid index in our handle array but will produce an
     * error in almost any program....thus we're ok if the open fails but the
     * caller ignores the returned error code */
    *pUsrHandle = MAXHANDLES;

    /* run through the handle array and look for a free handle */
    B_TRY(BestFirstFreeHandleGet(&handle));

handle_label:

    handle_array[handle].is_reserved = 0;

    if (task == B_OPEN_STANDARD_RESERVED)
    {
      /* task = B_OPEN_STANDARD; */
      task = B_OPEN_LICENSE; /* HL, 25-2-2000 changed this for SVP use */
    }

    /* Note; we MUST re-initialize prior to continuing so that a previously
     * interrupted open cannot leave us in an unknown state. */
    handle_array[handle] = handle_null;

    if (task == B_OPEN_RESERVE)
    {
      handle_array[handle].is_reserved = 1;
      *pUsrHandle = handle;
      B_ERRETURN(B_E_OK);
    }

    /* these may change depending on the port */
    handle_array[handle].port = port;
    handle_array[handle].entered_port = portnum;

    switch (port)
    {
    case B_PORT_OFFLINE:
      /* portnum codes hardware and license in case of OFFLINE port */
      OsHandle = (b_portnumtype)((b_int64) portnum);   			/* Doing this to make cl stop complaining */
      break;

    case B_PORT_RS232:
      OsHandle = BestOpenCOMInitial((int) portnum, 9600UL);
      B_TRY_FAIL(INVALID_OS_HANDLE == OsHandle ? B_E_RS232_OPEN : B_E_OK);
      break;

    case B_PORT_PCI_CONF:
      B_TRY(BestOpenPCI((int) portnum, &OsHandle));
      B_TRY_FAIL(INVALID_OS_HANDLE == OsHandle ? B_E_PCI_OPEN : B_E_OK);
      break;

    case B_PORT_PCI_IO:
      B_TRY(BestOpenIO((int) portnum, &OsHandle));
      B_TRY_FAIL(INVALID_OS_HANDLE == OsHandle ? B_E_IO_OPEN : B_E_OK);
      handle_array[handle].param = GlobalBaseAddr;
      break;

    case B_PORT_FASTHIF:
      B_TRY(BestOpenHIF((b_int32) portnum, &HPSlotId, &OsHandle));
      B_TRY_FAIL(INVALID_OS_HANDLE == OsHandle ? B_E_HIF_OPEN : B_E_OK);
      handle_array[handle].entered_port = HPSlotId;
      break;

    default:
      assert(0);
    }                           /*lint !e788 not all enums used */

    /* this is the ONLY place in the entire API where we "open" */
    handle_array[handle].portnumber = OsHandle;
    handle_array[handle].is_open = 1;

    /* set timeouts...must be done before any reads/writes
     * from this point on a BestClose() is required to release the handle 
     */

    if(B_OPEN_QUICKCHECK == task)
    {
      /* quickcheck open uses minimum timeouts.  
       * we're pretty close to the limit...reduce at your own risk !
       */
      BESTTIMEOUTS tmrStruct;
      /* 0, 0, 100, 0, 50 works for the 2926 but NOT the 2925.
       * The 2925 needs between 500 and 600 mS to respond!
       */
      BestSetCommToStruct(&tmrStruct, 0, 0, 700, 0, 50);
      B_TRY_PROGRESS(BestPortTimeoutSet(handle, &tmrStruct, TIMEOUTS_SET));
      BestConnectTimeoutSet(handle, 100);
    }
    else
    {
      B_TRY_PROGRESS(BestPortTimeoutSet(handle, NULL, TIMEOUTS_SET_DEF));
      BestConnectTimeoutDefSet(handle);
    }

    /* check connect/disconnect */
    B_TRY(BestConnect(handle));
    B_TRY(BestDisconnect(handle));

    /* identify hardware...this includes determination of the register set,
     * model, protocol, version etc. 
     */
    B_TRY(BestIdHardware(handle, task));

    /* don't bother with dynamic cap's if we're just checking.
     * this doesn't actually save a lot of time (the I/O is absolutely
     * the governing factor) but why do all those mallocs if we don't need to?
     */
    if(B_OPEN_QUICKCHECK != task)
    {
      /* create the dynamic capabilities */
      if (task != B_OPEN_BIOS_SWITCH)
      {
        B_TRY(BestCapaInit(handle));
      }

      /* copy the license code, so we know what license we have and what not */
      if ((task == B_OPEN_LICENSE) && (license_code))
      {
        for (j = 0; j < 3; ++j)
          license_code[j] = handle_array[handle].capable->capa_code[j];
      }
    }

    /* the ONLY successful open */
    *pUsrHandle = handle;
  }

  B_TRY_CATCH
  {
    B_TRY_PASSED
    {
      BestLastErrorStore(handle, B_TRY_RET);
      (void) BestClose(handle);
      *pUsrHandle = INVALID_BEST_HANDLE;

      /* Do not use B_ERRETRUN here!! Handleless error handling will get
       * confused. CZ */
      return (B_TRY_RET);
    }
  }

  /* just in case we changed this */
  if(B_OPEN_QUICKCHECK == task)
    BestConnectTimeoutDefSet(handle);

  B_ERRETURN(B_TRY_RET);
}

/*----------------------------------------------------------------------
 * BestReopen does an open/close on a stalled card
 *----------------------------------------------------------------------*/
b_errtype EXPORT
BestReopen(b_handletype handle, b_OpenType task)
{
  B_TRY_VARS_NO_PROG;

  B_TRY_BEGIN {
    B_TRY_HANDLE;

    /* unconditional disconnect */
    BestReleaseConnection(handle);

    /* establish new connection */
    B_TRY(BestOpenConnection(handle, connect_timeout_array[handle]));

  }

  B_ERRETURN (B_TRY_RET);
}

/* --------------------------------------------------------------------------
 * The default values for hardware identification.
 * Note that the handle[].hwinfo is reset in BestOpenAdv() and BestCloseAdv()
 * to B_HW_INFO_UNKNOWN
 * -------------------------------------------------------------------------- */

/* 
  the b_hwinfotype structure;

  b_int8ptr     product_string;
  b_hwtype      hw;             compatibility...old handle_array[].hw
  b_hwseriestype hwseries;      hardware series (no deep id)
  b_regfiletype regfile;        compatibility...old handle_array[].regfile
  b_int32 hwbitmask;            
 */

/****IMPORTANT******
 * Maintain these defines if you change product strings or add a hardware 
 * definition bit
 *******************/

#define MAX_PROD_STR_LEN	20

/* NOTE; These are deliberately defined here so that they cannot
 * be directly accessed outside of this file.
 * Use the set of functions defined below (search HARDWARE CHARACTERISTICS)
 * to set/check these hardware characteristics.
 */

#define B_HWDEF_HAS_FHIF      (0x01UL << 0)
#define B_HWDEF_HAS_ICHI      (0x01UL << 1)
#define B_HWDEF_IS_DEEP       (0x01UL << 2)
#define B_HWDEF_IS_CORE       (0x01UL << 3)
#define B_HWDEF_IS_BLKMODE    (0x01UL << 4)
#define B_HWDEF_HAS_PERF      (0x01UL << 5)
#define B_HWDEF_IS_COMPACT    (0x01UL << 6)
#define B_HWDEF_HAS_ICHI20    (0x01UL << 7)


/* just to save a little space */
#define B_HWDEF_HAS_FHIF_ICHI_PERF  (B_HWDEF_HAS_FHIF | B_HWDEF_HAS_ICHI | B_HWDEF_HAS_PERF)

static const b_hwinfotype hw_info_defs[] =
{
  /* E2925A ***MUST*** be the first entry */
  {"E2925A",      B_HW_E2925A,      B_SERIES_E2925A, PROTOCOL_1x, 
                  0x2925, 
                  0},

  {"E2925A_DEEP", B_HW_E2925A_DEEP, B_SERIES_E2925A, PROTOCOL_1x, 
                  0x2926, 
                  B_HWDEF_IS_DEEP | B_HWDEF_HAS_PERF},

  {"E2926A",      B_HW_E2926A,      B_SERIES_E2926A, PROTOCOL_2x, 
                  0x2926, 
                  B_HWDEF_HAS_FHIF_ICHI_PERF},

  {"E2926A_DEEP", B_HW_E2926A_DEEP, B_SERIES_E2926A, PROTOCOL_2x, 
                  0x2926, 
                  B_HWDEF_HAS_FHIF_ICHI_PERF | B_HWDEF_IS_DEEP},

  {"E2926B",      B_HW_E2926B,      B_SERIES_E2926A, PROTOCOL_2x, 
                  0x2926, 
                  B_HWDEF_HAS_FHIF_ICHI_PERF},

  {"E2926B_DEEP", B_HW_E2926B_DEEP, B_SERIES_E2926A, PROTOCOL_2x, 
                  0x2926, 
                  B_HWDEF_HAS_FHIF_ICHI_PERF | B_HWDEF_IS_DEEP},

  {"E2927A",      B_HW_E2927A,      B_SERIES_E2927A, PROTOCOL_2x, 
                  0x2927, 
                  B_HWDEF_HAS_FHIF_ICHI_PERF},

  {"E2927A_DEEP", B_HW_E2927A_DEEP, B_SERIES_E2927A, PROTOCOL_2x, 
                  0x2927, 
                  B_HWDEF_HAS_FHIF_ICHI_PERF | B_HWDEF_IS_DEEP},

  {"E2928A",      B_HW_E2928A,      B_SERIES_E2928A, PROTOCOL_2x, 
                  0x2928, 
                  B_HWDEF_HAS_FHIF_ICHI_PERF | B_HWDEF_HAS_ICHI20},

  {"E2928A_DEEP", B_HW_E2928A_DEEP, B_SERIES_E2928A, PROTOCOL_2x, 
                  0x2928, 
                  B_HWDEF_HAS_FHIF_ICHI_PERF | B_HWDEF_IS_DEEP| 
				  B_HWDEF_HAS_ICHI20},

  {"E2924A",      B_HW_E2924A,      B_SERIES_E2924A, PROTOCOL_2x, 
                  0x2924, 
                  B_HWDEF_HAS_FHIF | B_HWDEF_HAS_ICHI},

  {"E2925B",      B_HW_E2925B,      B_SERIES_E2925B, PROTOCOL_2x, 
                  0x2925, 
                  B_HWDEF_HAS_FHIF_ICHI_PERF},

  {"E2925B_DEEP", B_HW_E2925B_DEEP, B_SERIES_E2925B, PROTOCOL_2x, 
                  0x2925, 
                  B_HWDEF_HAS_FHIF_ICHI_PERF | B_HWDEF_IS_DEEP},

  {"E2940A",      B_HW_E2940A,      B_SERIES_E2940A, PROTOCOL_2x, 
                  0x2940, 
                  B_HWDEF_HAS_FHIF_ICHI_PERF | B_HWDEF_IS_COMPACT},

  {"E2940A_DEEP", B_HW_E2940A_DEEP, B_SERIES_E2940A, PROTOCOL_2x, 
                  0x2940, 
                  B_HWDEF_HAS_FHIF_ICHI_PERF | B_HWDEF_IS_COMPACT |
                  B_HWDEF_IS_DEEP },

  {"E2921A",      B_HW_E2921A,      B_SERIES_E2928A, PROTOCOL_2x, 
                  0x2921, B_HWDEF_HAS_ICHI20}


};

#define NUM_HW_DEFAULTS (int)(sizeof(hw_info_defs) / sizeof(b_hwinfotype))


/* --------------------------------------------------------------------------
 * Mainly for SVP...string list of available cards
 * -------------------------------------------------------------------------- */

b_errtype EXPORT BestGetCardTypeCount(size_t * pCount)
{
  *pCount = NUM_HW_DEFAULTS;
  return B_E_OK;
}

b_errtype EXPORT BestGetCardTypeAtIndex(
  b_param_stringlisttype * pCardList, 
  size_t idx
  )
{
  if ( idx  < NUM_HW_DEFAULTS )
  {
  	pCardList->name = hw_info_defs[idx].product_string;
  	pCardList->value = hw_info_defs[idx].hw;
  }
  else
  {
    assert(("Bad Index", 0));
    return B_E_PARAM_OUT_OF_RANGE;
  }

  return B_E_OK;
}

/* --------------------------------------------------------------------------
 * Some functions need to know if a PCI device id is actually a Best
 * -------------------------------------------------------------------------- */

b_bool EXPORT BestIsPciDeviceIdBest(b_int32 dev_id)
{
  int i;

  for(i = 0; i < NUM_HW_DEFAULTS; i++)
  {
    if(dev_id == hw_info_defs[i].dev_id)
      return 1;
  }
  return 0;
}


/* --------------------------------------------------------------------------
 * All hardware identification done here...b_hwinfotype declared in typedefs.h
 * -------------------------------------------------------------------------- */

/* In debug mode we still allow B_OPEN_FAILSAFE to bypass the version checks.
 * We also have a special task B_OPEN_BIOS_SWITCH which prepares for a
 * core-switch (and thus must skip the version check).
 */

#ifdef BEST_DEBUG
# define DO_VERSION_CHECK  \
          ((task != B_OPEN_FAILSAFE) && \
           (task != B_OPEN_BIOS_SWITCH) && \
           (task != B_OPEN_QUICKCHECK) \
          )
#else           /* BEST_DEBUG */
# define DO_VERSION_CHECK  \
          (task != B_OPEN_BIOS_SWITCH) && \
          (task != B_OPEN_QUICKCHECK)
#endif          /* BEST_DEBUG */


static b_errtype BestIdHardware(b_handletype handle, b_OpenType task)
{
  B_TRY_VARS_NO_PROG;
  b_handlestype *pHandle = &(handle_array[handle]);
  int i;
  int fIsOffline = (B_PORT_OFFLINE == pHandle->port);
  int fDoVerCheck = DO_VERSION_CHECK;
  char buf[30] = {0};
  b_int32 data;
  b_hwtype hw;
  b_errtype err;		/* for separate error checking */

  B_TRY_BEGIN
    {
      B_TRY(BestConnect(handle));

      /* classify hardware...currently B_HW_INFO_UNKNOWN */

      if (fIsOffline)
      {
	/* offline is special; the user combines the hardware and license into
	 * a mask ... put into handle_array[].portnumber in BestOpenAdv().
	 * We're only interested in hardware here. */

	hw = (b_hwtype) (B_CAPABILITY_HARDWARE_MASK & (b_int64) pHandle->portnumber);

	for (i = 0; i < NUM_HW_DEFAULTS; i++)
	{
	  if (hw == hw_info_defs[i].hw)
	  {
	    /* good one !! */
	    pHandle->hwinfo = hw_info_defs[i];
	    break;
	  }
	}
      }
      else
      {
	/* get HW type from board ... note that PRODUCT_STRING (0x02 in 2926
	 * mode !!) is byte-size across all boards; 
	 * 2926-bios/core -> a real product string. 
	 * 2925-bios -> 0x02 (BOARD_RSTMODE) returns an invalid
	 * string (no error) 
	 * 2925-core -> a "FunDummy" returns B_E_FUNC. 
	 * we can thus determine which card (and mode) we're talking to. 
	 */

	pHandle->regindex = REG_INDX_2926;

	/* an error here is either a real problem 
	 * or  we're in core-mode on a 2925 
	 */
	err = BestBasicBlockRead(handle, PRODUCT_STRING,
				 (b_int8ptr) buf, (b_int32) MAX_PROD_STR_LEN);


	if (err == B_E_OK)
	{
	  /* load the hardware struct corresponding to the product string */
	  for (i = 0; i < NUM_HW_DEFAULTS; i++)
	  {
	    if (0 == strcmp(buf, hw_info_defs[i].product_string))
	    {
	      /* good one !! */
	      pHandle->hwinfo = hw_info_defs[i];
	      goto hwfound_label;
	    }
	  }
	}
	else if (err != B_EFW_FUNC && err != B_E_FUNC) /* func error is reported by 25 in core mode */
	{
	  B_TRY_FAIL(err);
	}
      
	/* no product string found, try again for E2925A */
	pHandle->regindex = REG_INDX_2925;

	/* BUS_ADDR_LO does not exist in core and returns B_E_FUNC from there */
	switch (BestBasicBlockRead(handle, BUS_ADDR_LO, (b_int8ptr) & data, 4UL))
	{
	 case B_E_OK:
	  /* maybe it's a 25? */
	  B_TRY(BestBasicBlockRead(handle, PRODUCT_STRING,
				   (b_int8ptr) buf, (b_int32) MAX_PROD_STR_LEN));

	  /* search again for product string (same code as above) */
	  for (i = 0; i < NUM_HW_DEFAULTS; i++)
	  {
	    if (0 == strcmp(buf, hw_info_defs[i].product_string))
	    {
	      /* good one !! */
	      pHandle->hwinfo = hw_info_defs[i];
	      goto hwfound_label;
	    }
	  }

	  break;

	 case B_E_FUNC:
	 case B_EFW_FUNC:
	  /* we are not definitely sure but we assume a E2925A core */
	  BestLastErrorParamSet(handle, B_ERRPAR_1, (b_int32) (handle_array[handle].port));
	  B_TRY_FAIL(task == B_OPEN_BIOS_SWITCH ? B_E_OK : B_E_CANNOT_CONNECT_CORE);
	  pHandle->hwinfo = hw_info_defs[0];
	  pHandle->hwinfo.hwbitmask |= B_HWDEF_IS_CORE;
	  goto hwfound_label;

	 default:
	  BestLastErrorParamSet(handle, B_ERRPAR_1, (b_int32) B_EERR_UNKNOWN_REPLY);
	  B_TRY_ERROR(B_E_ERROR);
	}                         /*lint !e788 ... the rest of the errors */

	BestLastErrorParamSet(handle, B_ERRPAR_1, (b_int32) (handle_array[handle].port));
	B_TRY_ERROR(B_E_UNKNOWN_HARDWARE);
      }

    hwfound_label:    /* we found the hardware */

      /* take care of EVERYTHING else that is hardware dependent or make any
       * modifications to handle_array[].hwinfo here... */

      pHandle->regindex = (pHandle->hwinfo.regfile == PROTOCOL_1x) ?
	REG_INDX_2925 : REG_INDX_2926;

      /* we're done if we're working offline */
      if (fIsOffline)
	return B_E_OK;

      /* Some software components need to know if the board has been reset.
       * Thus, as part of every open we write the RESET_DETECT_VALUE into the
       * dummy register....it's automatically cleared if the board resets. */

      data = RESET_DETECT_VALUE;
      B_TRY(BestDummySet(handle, data));

      /* read back our RESET_DETECT_VALUE */
      data = 0UL;
      B_TRY(BestDummyGet(handle, &data));

      if (data != RESET_DETECT_VALUE)
      {
	BestLastErrorParamSet(handle, B_ERRPAR_1, (b_int32) (handle_array[handle].port));
	B_TRY_ERROR(B_E_UNKNOWN_HARDWARE);
      }

      /* ok, connection seems to work */

      if (!BestIs2925(handle))
      {
	B_TRY(BestBasicBlockRead(handle, VERSION_CODE, (b_int8ptr) buf, 4UL));

	switch (buf[0])
	{
	 case 'c':
	  /* core mode */
	  BestLastErrorParamSet(handle, B_ERRPAR_1,
				(b_int32) (handle_array[handle].port));
	  B_TRY_FAIL(task == B_OPEN_BIOS_SWITCH ? B_E_OK : B_E_CANNOT_CONNECT_CORE);
	  pHandle->hwinfo.regfile = PROTOCOL_1x;
	  pHandle->hwinfo.hwbitmask &= ~B_HWDEF_HAS_FHIF;
	  pHandle->hwinfo.hwbitmask |= B_HWDEF_IS_CORE;
	  break;

	 default:
	  /* already set up for E2926A, nothing to do */
	  /* This is the 26 way to detect reset, call it once to synchronize
	     data */
	  /* HL, 14.Jan 1999 commented out this call, why was it ever put in?*/
	  /* B_TRY (BestBasicIsBoardReset (handle, &data)); */
	  break;
	}
      }

      if (fDoVerCheck)
      {
	char s[256];
	/* get version string from bios and check that FW==CAPI versions */
	B_TRY(BestBasicBlockRead(handle, BIOS_STRING, (b_int8ptr) buf, 15UL));
	strcpy(s, BestIs2925(handle) ? bios25_version_string : bios26_version_string);
	strcat(s, "@");
	strcat(s, buf);
	BestLastErrorParamStringSet(handle, s);
	BestLastErrorParamSet(handle, B_ERRPAR_1, (b_int32) B_VERCHK_FIRMWARE);
	B_TRY_FAIL(strcmp(buf, 
			  (BestIs2925(handle) ? bios25_version_string : bios26_version_string)) ?
		   B_E_VERSION_MISMATCH : B_E_OK);
      }

      (void) BestDisconnect(handle);
    }

  B_ERRETURN(B_TRY_RET);
}


/* --------------------------------------------------------------------------
 * close the connection to the board and throw away all information
 * -------------------------------------------------------------------------- */

b_errtype EXPORT BestClose(b_handletype handle)
{
  B_ERRETURN(BestCloseAdv(handle, B_CLOSE_STANDARD));
}


b_errtype EXPORT BestCloseAdv(b_handletype handle, b_closetype task)
{
  B_TRY_VARS_NO_PROG;
  b_porttype port;
  b_portnumtype os_handle;
  B_TRY_BEGIN
  {
    B_TRY_HANDLE;
    B_TRY(BestCapaClose(handle)); /* delete the dynamic capabilities */
    (void) BestGetPortFromHandle(handle, &port);

    if (BestIs2925(handle))
    {
      if (port == B_PORT_RS232)
      {
        (void) BestConnect(handle);
        (void) BestRS232BaudRateSet(handle, 9600UL);
      }
    }

    /* save os handle for later use... */
    os_handle = handle_array[handle].portnumber;

    /* unconditional disconnect */
    BestReleaseConnection(handle);

    /* Total reset...this must be the last thing we do here. Note that this
     * is the only place that we "disconnect" a 2926 and is therefore
     * essential */

    handle_array[handle] = handle_null;

    /* well, except the reserved field */
    handle_array[handle].is_reserved = (b_int8) (task == B_CLOSE_RESERVED ? 1 : 0);

    /* the handle is now officially invalid */
    switch (port)
    {
    case B_PORT_OFFLINE:
      break;

    case B_PORT_RS232:
      BestCloseSerial(os_handle);
      break;

    case B_PORT_PCI_CONF:
    case B_PORT_PCI_IO:
      (void) BestClosePCI(os_handle);
      break;

    case B_PORT_FASTHIF:
      (void) BestCloseHIF(os_handle);
      break;

    default:
      break;
    }                           /*lint !e788 not all enums used */
  }

  B_ERRETURN(B_TRY_RET);
}


/* -------------------------------------------------------------------------
 * Set the baudrate of the RS232 Interface to value baudrate
 * ------------------------------------------------------------------------- */
b_errtype EXPORT BestRS232BaudRateSet(
    b_handletype handle,
    b_int32 baudrate
)
{
  B_TRY_VARS_NO_PROG;
  b_portnumtype OsHandle = INVALID_OS_HANDLE;
  int exclusive;
  B_TRY_BEGIN
  {
    B_TRY_FAIL(handle_array[handle].port == B_PORT_RS232 ?
      B_E_OK : B_E_WRONG_PORT);
    B_TRY_FAIL(BestIsHandleConnected(handle) ? B_E_OK : B_E_NOT_CONNECTED);
    exclusive = (handle_array[handle].is_connected == CONNECT_EXCLUSIVE);

    if (Best16BitRegisterFile(handle))
    {
      b_int8 inbuf[4];
      (void) BestLong2Stream(inbuf, &baudrate, 1UL);
      B_TRY(BestBasicCommand(handle, CMD_SER_BAUDRATE_SET,
          inbuf, IN_SER_BAUDRATE_SET,
          NULL, NULL));
    }

    else
    {
      /* set new baudrate on Mini */
      /* TODO: in error returned B_E_BAUDRATE */
      B_TRY(BestBasicBlockWrite(handle,
          RS232_BAUDRATE, (b_int8ptr) & baudrate, 4UL));
    }

    /* set new baudrate on host */
    handle_array[handle].param = baudrate;  /* for later queries only */

    if (Best16BitRegisterFile (handle)) {
      BestSerReleaseConnection (handle_array [handle].portnumber);
      B_TRY (BestWaitForClose (handle, 1000));
    }

    BestCloseSerial(handle_array[handle].portnumber);
    
    OsHandle = BestOpenCOMInitial((int) handle_array[handle].entered_port,
      baudrate);

    B_TRY_FAIL(OsHandle == INVALID_OS_HANDLE ? B_E_BAUDRATE : B_E_OK);
    handle_array[handle].portnumber = OsHandle;

    /* You MUST set the default timeouts...Win95 returns to "No Timeouts"
     * after a close !! */
    B_TRY(BestPortTimeoutSet(handle, NULL, TIMEOUTS_SET_DEF));

    if (BestIs2925(handle))
    {
      (void) BestDisconnect(handle);  /* to get effect on Mini */

      if (exclusive)
      {
        B_TRY(BestConnectEx(handle));
      }

      else
      {
        B_TRY(BestConnect(handle));
      }
    }
    else
    {
      B_TRY(BestOpenConnection(handle, connect_timeout_array[handle]));
    }
  }

  return B_TRY_RET;
}


/* **********************************************************************
   handle related functions (not in firmware)
   ********************************************************************** */

b_errtype EXPORT BestDebug(b_handletype handle, b_int32 debug)
{
  B_DECLARE_FUNCNAME("BestDebug");

  B_TRY_VARS_NO_PROG;

  B_TRY_BEGIN
  {
    B_TRY_FCT_PARAM(1, debug > 0x1);
    B_TRY_HANDLE;

    if (debug != 0)
    {
      handle_array[handle].cb_printf = handle_array[handle].cb_p_backup;
    }
    else
    {
      handle_array[handle].cb_printf = NULL;
    }
  }

  B_ERRETURN(B_TRY_RET);
}


b_errtype EXPORT BestGetDebugOutFromHandle(b_handletype handle,
    void (**std_out) (char *,...))
{
  HANDLECHECK;
  *std_out = handle_array[handle].cb_printf;
  B_ERRETURN(B_E_OK);
}


b_errtype EXPORT BestGetPortFromHandle(b_handletype handle,
    b_porttype * port)
{
  HANDLECHECK;
  *port = handle_array[handle].port;
  B_ERRETURN(B_E_OK);
}


b_errtype EXPORT BestGetPortnumberFromHandle(b_handletype handle,
    b_int32 * portnumber)
{
  HANDLECHECK;
  *portnumber = handle_array[handle].entered_port;
  B_ERRETURN(B_E_OK);
}

b_errtype EXPORT BestGetHWFromHandle(b_handletype handle,
    b_hwtype * hw)
{
  HANDLECHECK;
  * hw = handle_array[handle].hwinfo.hw;
  B_ERRETURN(B_E_OK);
}

b_errtype EXPORT BestGetProductstringFromHandle(b_handletype handle,
						b_charptrtype * pChar)
{
  HANDLECHECK;
  assert (pChar != NULL);
  *pChar = handle_array[handle].hwinfo.product_string;
  B_ERRETURN(B_E_OK);
}

b_errtype EXPORT BestGetPciDevIdFromHandle(b_handletype handle,
    b_int32 * dev_id)
{
  HANDLECHECK
  * dev_id = handle_array[handle].hwinfo.dev_id;
  B_ERRETURN(B_E_OK);
}


/* --------------------------------------------------------------------------
 * HARDWARE CHARACTERISTICS
 * The following functions are used to distinguish hardware.
 * Because these functions are often called before any handle checks it
 * is possible that we have a bad handle but we don't have a b_errtype to
 * return....therefore we return the value which is most likely to make
 * the calling function continue and call a routine which will trap the
 * handle error.
 * -------------------------------------------------------------------------- */

b_bool EXPORT BestIsOpen(b_handletype handle)
{
  /* HL; return FALSE (!!!!) if this is a bad handle */
  return (b_bool) (__HANDLECHECK ? 1 : 0 ); 
}

b_bool EXPORT BestIsKnown(b_handletype handle)
{
  /* CZ; return FALSE (!!!!) if this is a bad handle */
  return (b_bool) (__HANDLECHECK ? 
    handle_array[handle].hwinfo.hwseries != B_SERIES_UNKNOWN : 0);
}

b_bool EXPORT BestIs2925(b_handletype handle)
{
  /* SCR; return TRUE (!!!!) if this is a bad handle */
  return (b_bool) (__HANDLECHECK ? 
    handle_array[handle].hwinfo.hwseries == B_SERIES_E2925A : 1);
}


b_bool EXPORT BestIs2925Deep(b_handletype handle)
{
  /* SCR; return TRUE (!!!!) if this is a bad handle */
  return (b_bool) (__HANDLECHECK ? 
    handle_array[handle].hwinfo.hw == B_HW_E2925A_DEEP : 1);
}


b_bool EXPORT BestHasFastHost(b_handletype handle)
{
  /* SCR; return FALSE if this is a bad handle */
  return (b_bool) (__HANDLECHECK ? 
    (handle_array[handle].hwinfo.hwbitmask & B_HWDEF_HAS_FHIF) != 0 : 0);
}


b_bool EXPORT BestHasIchiban(b_handletype handle)
{
  /* SCR; return FALSE if this is a bad handle */
  return (b_bool) (__HANDLECHECK ? 
    (handle_array[handle].hwinfo.hwbitmask & B_HWDEF_HAS_ICHI) != 0 : 0);
}

b_bool EXPORT BestHasPerformance(b_handletype handle)
{
  /* SCR; return FALSE if this is a bad handle */
  return (b_bool) (__HANDLECHECK ? 
    (handle_array[handle].hwinfo.hwbitmask & B_HWDEF_HAS_PERF) != 0 : 0);
}

b_bool EXPORT BestIsDeep(b_handletype handle)
{
  /* SCR; return FALSE if this is a bad handle */
  return (b_bool) (__HANDLECHECK ? 
    (handle_array[handle].hwinfo.hwbitmask & B_HWDEF_IS_DEEP) != 0 : 0);
}

b_bool EXPORT BestIsCompact(b_handletype handle)
{
  return (b_bool) (__HANDLECHECK ? 
    (handle_array[handle].hwinfo.hwbitmask & B_HWDEF_IS_COMPACT) != 0 : 0);
}

b_bool EXPORT BestIsCore(b_handletype handle)
{
  return (b_bool) (__HANDLECHECK ? 
    (handle_array[handle].hwinfo.hwbitmask & B_HWDEF_IS_CORE) != 0 : 0);
}


b_bool EXPORT BestIsInBlockMode(b_handletype handle)
{
  return (b_bool) (__HANDLECHECK ? 
    (handle_array[handle].hwinfo.hwbitmask & B_HWDEF_IS_BLKMODE) != 0 : 0);
}

void BestHwBlockModeBitSet(b_handletype handle, b_bool fSet)
{
  BestHwBitMaskSet(handle, B_HWDEF_IS_BLKMODE, fSet);
}

void BestHwBitMaskSet(b_handletype handle, b_int32 mask, b_bool fSet)
{
  if(__HANDLECHECK)
  {
    if(fSet)
      handle_array[handle].hwinfo.hwbitmask |= mask;
    else
      handle_array[handle].hwinfo.hwbitmask &= ~mask;
  }

  return;
}

b_bool Best16BitRegisterFile(b_handletype handle)
{
  /* SCR; return false if this is a bad handle */
  return (b_bool) (__HANDLECHECK ? 
    handle_array[handle].hwinfo.regfile == PROTOCOL_2x : 0);
}

b_hwseriestype EXPORT BestHwSeriesGet(b_handletype handle)
{
  return (__HANDLECHECK ? 
    handle_array[handle].hwinfo.hwseries : B_SERIES_UNKNOWN);
}

/* --------------------------------------------------------------------------
 * SCR; 29.10.97; Added the new I/O scheme...see iocommon.c
 * Note; 2926's must still call this AT LEAST ONCE during an Open.
 * -------------------------------------------------------------------------- */

b_errtype EXPORT BestConnect(b_handletype handle)
{
  B_TRY_VARS_NO_PROG;

  HANDLECHECK;

  B_TRY_BEGIN
  {
    /* real connects are only done for the 2925 */
    if (BestIs2925(handle) || !BestIsKnown(handle))
    {
      B_TRY(BestOpenConnection(handle, connect_timeout_array[handle]));
    }

    handle_array[handle].is_connected = CONNECT_NORMAL;
  }

  B_TRY_CATCH
  {
    /* SCR; 11.97; BestReleaseConnection() now waits for close */
    BestReleaseConnection(handle);
  }

  B_ERRETURN(B_TRY_RET);
}



/* --------------------------------------------------------------------------
 * TODO; This may be an obsolete function....where used?
 * -------------------------------------------------------------------------- */

b_errtype EXPORT BestConnectEx(b_handletype handle)
{
  B_TRY_VARS;
  b_int8 zw;
  volatile int was_connected = 0;
  /* TODO ... this was done to allow old code and OFFLINE to work ! What do
   * we really want to do with 16BitRegisterFile ? */

  B_TRY_BEGIN
  {
    if (BestIs2925(handle) || !BestIsKnown(handle))
    {
      was_connected = BestIsHandleConnected(handle);
      B_TRY_PROGRESS(BestConnect(handle));

      if (B_PORT_OFFLINE != handle_array[handle].port)
      {
        zw = LOCK_BOARD;
        B_TRY(BestBasicBlockWrite(handle, SERVICE_CMD, &zw, 1UL));
      }

      handle_array[handle].is_connected = CONNECT_EXCLUSIVE;
    }
  }

  B_TRY_CATCH
  {
    B_TRY_RET = B_E_CANNOT_CONNECT_EXCLUSIVE;
    BestLastErrorParamSet(handle, B_ERRPAR_1,
      (b_int32) (handle_array[handle].port));

    B_TRY_PASSED
    {
      /* nothing to do */
    }

    B_TRY_FAILED
    {
      was_connected = 0;
    }

    if (!was_connected)
      (void) BestDisconnect(handle);
  }

  B_ERRETURN(B_TRY_RET);
}



/* --------------------------------------------------------------------------
 * SCR; Note that a 2926 will NEVER DISCONNECT until closed.
 * Also note that we check for an open handle (even in 2926 mode) just to
 * preserve the logic.
 * -------------------------------------------------------------------------- */

b_errtype EXPORT BestDisconnect(b_handletype handle)
{
  /* here for debugging purposes on the 2926 */
  HANDLECHECK;

  if (BestIs2925(handle) || !BestIsKnown(handle))
  {
    /* SCR; 11.97; BestReleaseConnection() now waits for close */
    BestReleaseConnection(handle);
    handle_array[handle].is_connected = 0;
  }

  B_ERRETURN(B_E_OK);
}


b_errtype EXPORT BestDummySet(b_handletype handle, b_int32 dummy)
{
  B_ERRETURN(BestBasicBlockWrite(handle, DUMMY_REGISTER, (b_int8ptr) & dummy, 4UL));
}


b_errtype EXPORT BestDummyGet(b_handletype handle, b_int32 * dummy)
{
  B_ERRETURN(BestBasicBlockRead(handle, DUMMY_REGISTER, (b_int8ptr) dummy, 4UL));
}


b_errtype EXPORT BestPing(b_handletype handle)
{
  B_TRY_VARS_NO_PROG;

  B_TRY_BEGIN
  {
    B_TRY_FAIL(BestIs2925(handle) ? B_E_NOT_E2925A : B_E_OK);
    B_TRY(BestBasicCommand(handle, CMD_PING, NULL, IN_PING, NULL, NULL));
  }

  B_ERRETURN(B_TRY_RET);
}


b_errtype EXPORT BestClockSet (b_handletype handle, b_int32 clkvalue)
{
  b_errtype	err;

  err = BBCSendLong (handle, CMD_CLOCK_SET, &clkvalue);
  B_ERRETURN (err);
}


b_errtype EXPORT BestClockGet (b_handletype handle, b_int32 *clkvalue)
{
  b_errtype	err;

  err = BBCReceiveLong (handle, CMD_CLOCK_GET, clkvalue);
  B_ERRETURN (err);
}


b_bool EXPORT BestHasIchiban20(b_handletype handle)
{
  /* SCR; return FALSE if this is a bad handle */
  return (b_bool) (__HANDLECHECK ? 
    (handle_array[handle].hwinfo.hwbitmask & B_HWDEF_HAS_ICHI20) != 0 : 0);
}

/* no firmware here */

#else

/* Firmware here */

BOOL BestHasIchiban20(b_handletype handle)
{
  UWORD value;
  value = ASIC_WORD_READ (CHIP_REVISION_REG);
  return(value==0x200); 
}

#endif /* firmware */


#ifndef E2921A_MIN

/* --------------------------------------------------------------------------
 * 2926 only  - locking functions (resources can get locked and unlocked)
 * ------------------------------------------------------------------------- */

static b_errtype resource2page(b_ccharptrtype pFuncName,
    b_handletype handle,
    b_resourcetype resource,
    b_int8ptr page)
{
  B_TRY_VARS_NO_PROG;

  B_TRY_BEGIN
  {
    B_TRY_FAIL(Best16BitRegisterFile(handle) ? B_E_OK : B_E_NOT_E2925A);
    pFuncName;
    handle;
    
    switch (resource) {
     case B_RESLOCK_MAILBOX:
      *page = CMDG_MAILBOX_LOCK;
      break;

     case B_RESLOCK_CPUPORT:
      *page = CMDG_CPUPORT_ACTION_LOCK;
      break;

     case B_RESLOCK_STATIC_IO:
      *page = CMDG_STATIO_ACTION_LOCK;
      break;

     case B_RESLOCK_EXERCISER:
      *page = CMDG_EXEFHIFACTION_LOCK;
      break;

     case B_RESLOCK_TRACEMEM:
      *page = CMDG_TRACEACTION_LOCK;
      break;

     case B_RESLOCK_OBSERVER:
      *page = CMDG_OBSACTION_LOCK;
      break;

     case B_RESLOCK_PERFORMANCE:
      *page = CMDG_PERFACTION_LOCK;
      break;

    case B_RESLOCK_SST:
      *page = CMDG_SSTACTION_LOCK;
      break;

     default:
      B_FCT_PARAM_RANGE_CHK(resource,
			    B_RESLOCK_PATT_TERM_0,
			    B_RESLOCK_PATT_TERM_20);
      *page = (b_int8) (CMDG_PATT_SET + resource - B_RESLOCK_PATT_TERM_0);
      break;
    }                           /*lint !e788 not all enums used */
  }

  return B_TRY_RET;
}


b_errtype EXPORT BestResourceLock(
    b_handletype handle,
    b_resourcetype resource)
{
  B_DECLARE_FUNCNAME("BestResourceLock [reslock]");
  B_TRY_VARS_NO_PROG;
  b_int8 page;
  /* note; param resource checked in the call to resource2page() */

  HANDLECHECK;

  B_TRY_BEGIN {
    B_TRY(resource2page(pFuncName, handle, resource, &page));
    B_TRY(BestBasicCommand(handle, CMD_MERGE(page, CMDM_LOCK),
			   NULL, 0, NULL, NULL));
  }

  B_ERRETURN(B_TRY_RET);
}


/* --------------------------------------------------------------------------
 * 2926 only -- unlock a specific resource
 * ------------------------------------------------------------------------- */

b_errtype EXPORT BestResourceUnlock(
    b_handletype handle,
    b_resourcetype resource)
{
  B_DECLARE_FUNCNAME("BestResourceUnlock [resunlock]");
  B_TRY_VARS_NO_PROG;
  b_int8 page;
  /* note; param resource checked in the call to resource2page() */

  HANDLECHECK;

  B_TRY_BEGIN {
    B_TRY (resource2page (pFuncName, handle, resource, &page));
    B_TRY (BestBasicCommand (handle, CMD_MERGE(page, CMDM_UNLOCK),
			     NULL, 0, NULL, NULL));
  }

  B_ERRETURN(B_TRY_RET);
}


/* -------------------------------------------------------------------------
 * unlock all resources immediately (this is for the case that someone forgot
 * to unlock a resource and you need to talk to the card)
 * ------------------------------------------------------------------------- */
b_errtype EXPORT BestAllResourceUnlock(b_handletype handle)
{
  B_DECLARE_FUNCNAME ("BestAllResourceUnlock [allresunlock]");
  B_ERRETURN(BestBasicCommand(handle, CMD_UNLOCK_ALL, NULL, 0, NULL, NULL));
}


/* -------------------------------------------------------------------------
 * check if a resource is locked
 * ------------------------------------------------------------------------- */

b_errtype EXPORT BestResourceIsLocked(
    b_handletype handle,
    b_resourcetype resource,
    b_int32 * lock_count,
    b_porttype * lock_port)
{
  B_DECLARE_FUNCNAME("BestResourceIsLocked [islocked]");
  B_TRY_VARS_NO_PROG;
  b_int8 page;
  b_int8 outbuf[OUT_ISLOCKED];
  b_int8ptr outptr = outbuf;
  b_int16 outsize = OUT_ISLOCKED;
  b_int32 portval;
  /* note; param resource checked in the call to resource2page() */

  HANDLECHECK;
  B_FCT_PARAM_NULL_POINTER_CHK (lock_count);
  B_FCT_PARAM_NULL_POINTER_CHK (lock_port);

  B_TRY_BEGIN {
    *lock_count = 0;
    *lock_port = B_PORT_CURRENT;

    B_TRY(resource2page(pFuncName, handle, resource, &page));
    B_TRY(BestBasicCommand(handle, CMD_ISLOCKED,
			   &page, IN_ISLOCKED,
			   outptr, &outsize));
    outptr = BestStream2Long(lock_count, outptr, 1UL);
    outptr = BestStream2Long(&portval, outptr, 1UL);

    if (*lock_count) {
      switch (portval) {
       case PORTID_SERIAL:
	*lock_port = B_PORT_RS232;
	break;

       case PORTID_PARALLEL:
	*lock_port = B_PORT_FASTHIF;
	break;

       case PORTID_PCI:
	*lock_port = B_PORT_PCI_CONF;
	break;

       default:
	*lock_port = B_PORT_OFFLINE;
	break;
      }

#ifndef BEST_FIRMWARE
      switch (handle_array[handle].port) {
       case B_PORT_RS232:
	if (*lock_port == B_PORT_RS232)
	  *lock_port = B_PORT_CURRENT;
	
	break;

       case B_PORT_FASTHIF:
	if (*lock_port == B_PORT_FASTHIF)
	  *lock_port = B_PORT_CURRENT;
	
	break;

       case B_PORT_PCI_CONF:
       case B_PORT_PCI_IO:
	if (*lock_port == B_PORT_PCI_CONF)
	  *lock_port = B_PORT_CURRENT;

          break;
      }                       /*lint !e787 not all enums used */

#endif
      
    }
  }

  B_ERRETURN(B_TRY_RET);
}


#endif  /* E2921A_MIN */


b_errtype BestAbstractPropByteSet(b_handletype handle,
				  b_int16 cmdcode,
				  b_int8 prpid,
				  b_int32 prpval)
{
  B_DECLARE_FUNCNAME("BestAbstractPropByteSet");

  b_errtype err;
  b_int8 buf[2];
  b_int8ptr bufptr = buf;
  b_int8 value = (b_int8) prpval;
  B_FCT_PARAM_CHK(3, prpval > 0xff);

  bufptr = BestByteCopy(bufptr, &prpid, 1UL);
  bufptr = BestByteCopy(bufptr, &value, 1UL);
  err = BestBasicCommand(handle, cmdcode, buf, 2, NULL, NULL);

  return err;
}


b_errtype BestAbstractPropShortSet(b_handletype handle,
    b_int16 cmdcode,
    b_int8 prpid,
    b_int32 prpval)
{
  B_DECLARE_FUNCNAME("BestAbstractPropShortSet");

  b_errtype err;
  b_int8 buf[3];
  b_int8ptr bufptr = buf;
  b_int16 value = (b_int16) prpval;
  B_FCT_PARAM_CHK(3, prpval > 0xffff);

  bufptr = BestByteCopy(bufptr, &prpid, 1UL);
  bufptr = BestWord2Stream(bufptr, &value, 1UL);
  err = BestBasicCommand(handle, cmdcode, buf, 3, NULL, NULL);

  return err;
}


b_errtype BestAbstractPropLongSet(b_handletype handle,
    b_int16 cmdcode,
    b_int8 prpid,
    b_int32 prpval)
{
  b_errtype err;
  b_int8 buf[5];
  b_int8ptr bufptr = buf;
  bufptr = BestByteCopy(bufptr, &prpid, 1UL);
  bufptr = BestLong2Stream(bufptr, &prpval, 1UL);
  err = BestBasicCommand(handle, cmdcode, buf, 5, NULL, NULL);

  return err;
}


b_errtype BestAbstractPropByteGet(b_handletype handle,
    b_int16 cmdcode,
    b_int8 prpid,
    b_int32 * prpval)
{
  b_errtype err;
  b_int8 outbuf[1];
  b_int16 outsize = 1;
  err = BestBasicCommand(handle, cmdcode, &prpid, 1, outbuf, &outsize);
  *prpval = outbuf[0];
  return err;
}


b_errtype BestAbstractPropShortGet(b_handletype handle,
    b_int16 cmdcode,
    b_int8 prpid,
    b_int32 * prpval)
{
  b_errtype err;
  b_int8 outbuf[2];
  b_int16 outsize = 2;
  b_int16 value;
  err = BestBasicCommand(handle, cmdcode, &prpid, 1, outbuf, &outsize);
  (void) BestStream2Word(&value, outbuf, 1UL);
  *prpval = value;
  return err;
}


b_errtype BestAbstractPropLongGet(b_handletype handle,
    b_int16 cmdcode,
    b_int8 prpid,
    b_int32 * prpval)
{
  b_errtype err;
  b_int8 outbuf[4];
  b_int16 outsize = 4;
  err = BestBasicCommand(handle, cmdcode, &prpid, 1, outbuf, &outsize);
  (void) BestStream2Long(prpval, outbuf, 1UL);
  return err;
}


/* --------------------------------------------------------------------------
 * SCR; this series of helper functions simplify calls to BestBasicCommand
 * which send OR receive (NOT both) simple variables.
 * NOTE; the IN_/OUT_ macros are not used....the stream lengths are implicit.
 * -------------------------------------------------------------------------- */

/* SENDING simple vars...OUT_ macro must be 0 !!! */
b_errtype BBCSendByte(b_handletype handle, b_int16 cmdcode, b_int8ptr pByte)
{
  b_int8 cmdbuf[sizeof(b_int8) + 1];
  (void) BestByteCopy(cmdbuf, pByte, 1UL);
  return BestBasicCommand(handle, cmdcode, cmdbuf, sizeof(b_int8), NULL, NULL);
}

b_errtype BBCSendWord(b_handletype handle, b_int16 cmdcode, b_int16 * pUWord)
{
  b_int8 cmdbuf[sizeof(b_int16) + 1];
  (void) BestWord2Stream(cmdbuf, pUWord, 1UL);
  return BestBasicCommand(handle, cmdcode, cmdbuf, sizeof(b_int16), NULL, NULL);
}

b_errtype BBCSendLong(b_handletype handle, b_int16 cmdcode, b_int32 * pULong)
{
  b_int8 cmdbuf[sizeof(b_int32) + 1];
  (void) BestLong2Stream(cmdbuf, pULong, 1UL);
  return BestBasicCommand(handle, cmdcode, cmdbuf, sizeof(b_int32), NULL, NULL);
}
/* RECEIVING simple vars */
b_errtype BBCReceiveByte(b_handletype handle, b_int16 cmdcode, b_int8ptr pByte)
{
  b_int8 cmdbuf[sizeof(b_int8) + 1] = {0};
  b_int16 outsize = sizeof(b_int8);
  b_errtype err = BestBasicCommand(handle, cmdcode, NULL, 0, cmdbuf, &outsize);
  (void) BestByteCopy(pByte, cmdbuf, 1UL);
  return err;
}

b_errtype BBCReceiveWord(b_handletype handle, b_int16 cmdcode, b_int16 * pUWord)
{
  b_int8 cmdbuf[sizeof(b_int16) + 1] = {0};
  b_int16 outsize = sizeof(b_int16);
  b_errtype err = BestBasicCommand(handle, cmdcode, NULL, 0, cmdbuf, &outsize);
  (void) BestStream2Word(pUWord, cmdbuf, 1UL);
  return err;
}

b_errtype BBCReceiveLong(b_handletype handle, b_int16 cmdcode, b_int32 * pULong)
{
  b_int8 cmdbuf[sizeof(b_int32) + 1] = {0};
  b_int16 outsize = sizeof(b_int32);
  b_errtype err = BestBasicCommand(handle, cmdcode, NULL, 0, cmdbuf, &outsize);
  (void) BestStream2Long(pULong, cmdbuf, 1UL);
  return err;
}


/*---------------------------------------------------------
* capi_version returns the version string
*---------------------------------------------------------*/
b_ccharptrtype EXPORT capi_version()
{
  return capi_version_string;
}

/*---------------------------------------------------------------------------*
 * void EXPORT BestCapiVersion (int verArray[4])
 *
 * Purpose	: get CAPI version number from DLL
 *---------------------------------------------------------------------------*/
void EXPORT BestCapiVersion (char verArray[4])
{
  const char lVerArray[4] = {capi_version_number};
  size_t i;

  for (i=0; i<4; i++)
  {
    verArray[i] = lVerArray[i];
  }
}

#ifndef BEST_FIRMWARE

/* --------------------------------------------------------------------------
 * Use these functions for ALL byte-ordering.
 * They are tested safe for (dest == src) on MSVC (Win32) and Borland 4.52
 * NOTE; memcpy IS NOT always safe for dest==src (compiler dependent) !!!
 * -------------------------------------------------------------------------- */

/* --------------------------------------------------------------------------
 * NOTE; Non-linear memory models will be limited i.e. DOS; size_t == UInt16
 * -------------------------------------------------------------------------- */


/******************************************************************************/
b_int8ptr EXPORT BestByteCopy(b_int8ptr dest, b_int8ptr src, b_int64 count)
{
  if (dest != src)
  {
    memcpy(dest, src, (size_t) count);
  }
  return (dest + (size_t) count);
}


/******************************************************************************/
typedef struct _WORD2STREAM
{
  b_int8 AsBytes[2];
} WORD2STREAM;


/******************************************************************************/
b_int8ptr EXPORT BestWord2Stream(b_int8ptr dest, b_int16ptr src, b_int32 count) /* num_words */
{
  WORD2STREAM tmp;
  if (BestIsHostLittleEndian())
  {
    while (count--)
    {
      tmp = *(WORD2STREAM *) src;
      *dest++ = tmp.AsBytes[1];
      *dest++ = tmp.AsBytes[0];
      src++;
    }
  }
  else if (dest != (b_int8ptr) src)
  {
    memcpy(dest, src, (size_t) (count * sizeof(b_int16)));
    dest += (size_t) (count * sizeof(b_int16));
  }

  return dest;
}


/******************************************************************************/
b_int8ptr EXPORT BestStream2Word(b_int16ptr dest, b_int8ptr src, b_int32 count) /* num_words */
{
  if (BestIsHostLittleEndian())
  {
    while (count--)
    {
      *dest = (b_int16) (((b_int16) src[0] << 8) | (b_int16) src[1]);
      dest++;
      src += 2;
    }
  }
  else if ((b_int8ptr) dest != src)
  {
    memcpy(dest, src, (size_t) (count * sizeof(b_int16)));
    src += (size_t) (count * sizeof(b_int16));
  }

  return src;
}


/******************************************************************************/
typedef struct _LONG2STREAM
{
  b_int8 AsBytes[4];
} LONG2STREAM;


/******************************************************************************/
b_int8ptr EXPORT BestLong2Stream(b_int8ptr dest, b_int32ptr src, b_int32 count) /* num_dwords */
{
  LONG2STREAM tmp;
  if (BestIsHostLittleEndian())
  {
    while (count--)
    {
      tmp = *(LONG2STREAM *) src;
      *dest++ = tmp.AsBytes[3];
      *dest++ = tmp.AsBytes[2];
      *dest++ = tmp.AsBytes[1];
      *dest++ = tmp.AsBytes[0];
      src++;
    }
  }
  else if (dest != (b_int8ptr) src)
  {
    memcpy(dest, src, (size_t) (count * sizeof(b_int32)));
    dest += (size_t) (count * sizeof(b_int32));
  }

  return dest;
}


/******************************************************************************/
b_int8ptr EXPORT BestStream2Long(b_int32ptr dest, b_int8ptr src, b_int32 count) /* num_dwords */
{
  if (BestIsHostLittleEndian())
  {
    while (count--)
    {
      *dest = ((b_int32) src[0] << 24) |
      ((b_int32) src[1] << 16) |
      ((b_int32) src[2] << 8) |
      (b_int32) (src[3]);
      dest++;
      src += 4;
    }
  }
  else if ((b_int8ptr) dest != src)
  {
    memcpy(dest, src, (size_t) (count * sizeof(b_int32)));
    src += (size_t) (count * sizeof(b_int32));
  }

  return src;
}

#endif /* not firmware */


/* TODO: Move this function to internal.c but remove the call from
   aprpdefload to have a linkable source code delivery. CZ */

/* ------------------------------------------------------------------------- *
 * This functions sets the debug properties on board of the e2926a. If the   *
 * hardware is not e2926, this call is ignored.                              *
 * ------------------------------------------------------------------------- */

b_errtype EXPORT BestDebugPropSet(b_handletype handle,
    b_debugtype prop,
    b_int32 value)
{
  B_DECLARE_FUNCNAME ("BestDebugPropSet [dbgprpset]"); 
  B_TRY_VARS_NO_PROG;

  B_TRY_BEGIN
  {
    if (!BestIs2925(handle))
    {
      /* this call is 2926 only */
      /* depending on the value of 'value' the debug property is either set
       * or not. Different commands are called. (see below). */
      if (prop == B_FW_ALL)
      {
        /* set or clear all is the command */
        B_TRY(BestBasicCommand(handle,
            (b_int16) (value ? CMD_DBG_SECTION_SETALL :
              CMD_DBG_SECTION_CLEARALL),
            NULL, 0, NULL, NULL));
      }

      else
      {
        b_int8 inbuf = (b_int8) prop;
        B_TRY(BestBasicCommand(handle,
            (b_int16) (value ? CMD_DBG_SECTION_SET :
              CMD_DBG_SECTION_CLEAR),
            &inbuf, 1, NULL, NULL));
      }
    }
  }


  B_ERRETURN(B_TRY_RET);
}
